	function hdlspec = hdlfirref(name, bitspec)
	%hdlspec = hdlfirref(name, bitspec)
	%
	%Generates FIR reference implementation. The reference implementation
	%is input-to-output true with the bit-level implementation.
	%
	%Arguments:
	%  name - entity name. The filename will be name.vhdl.
	%  bitspec - specification structure
	%
	%Returns:
	%  hdlspec - HDL specification structure
	
	%Copyright 2008, 2010 Anton Blad
	%
	%This file is part of firgen.
	%
	%firgen is free software: you can redistribute it and/or modify
	%it under the terms of the GNU General Public License as published by
	%the Free Software Foundation, either version 3 of the License, or
	%(at your option) any later version.
	%
	%firgen is distributed in the hope that it will be useful,
	%but WITHOUT ANY WARRANTY; without even the implied warranty of
	%MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	%GNU General Public License for more details.
	%
	%You should have received a copy of the GNU General Public License
	%along with firgen.  If not, see <http://www.gnu.org/licenses/>
	
	if bitspec.arith.signeddata == 1
		cmult = 'scmult';
		add = 'sadd';
	else
		cmult = 'ucmult';
		add = 'uadd';
	end
	
	hdlspec.top = name;
	hdlspec.components = [];
	hdlspec.components = [hdlspec.components, hdlcompiface(cmult)];
	hdlspec.components = [hdlspec.components, hdlcompiface(add)];
	hdlspec.components = [hdlspec.components, hdlcompiface('reg')];
	
	comp.iface.name = name;
	comp.iface.generics = [];
	
	for kin = 0:bitspec.numin-1
		comp.iface.inputs(kin+1) = hdlsignal(sprintf('x_%d', kin), 1, bitspec.arith.wdata);
	end
	for kout = 0:bitspec.numout-1
		comp.iface.outputs(kout+1) = hdlsignal(sprintf('y_%d', kout), 1, bitspec.arith.wout);
	end
	
	comp.arch.class = 2;
	comp.arch.components = {cmult, add, 'reg'};
	comp.arch.signals = struct('name', {}, 'type', {});
	comp.arch.instances = struct('type', {}, 'generics', {}, 'inputs', {}, 'outputs', {});
	
	ksig = 0;
	kinst = 0;
	kass = 0;
	
	for b = 0:bitspec.numout-1
		h = bitspec.branches{b+1}.h;
		inputdelays = fliplr(ceil((length(h)-(0:bitspec.numin-1))/bitspec.numin)-1);
		pipedelay = bitspec.branches{b+1}.pipedelay;
	
		% Generate delay line
		for p = 0:bitspec.numin-1
			for d = 0:inputdelays(p+1)
				ksig = ksig + 1;
				comp.arch.signals(ksig) = hdlsignal(sprintf('x_%d_%d_%d', b, p, d), 1, bitspec.arith.wdata);
			end
			kass = kass + 1;
			comp.arch.assignments(kass) = hdlass(sprintf('x_%d_%d_0', b, p), sprintf('x_%d', p));
			for d = 0:inputdelays(p+1)-1
				kinst = kinst + 1;
				comp.arch.instances(kinst) = hdlinst('reg', ...
					hdlmap('wordlength', bitspec.arith.wdata), ...
					hdlmap('d', sprintf('x_%d_%d_%d', b, p, d)), ...
					hdlmap('q', sprintf('x_%d_%d_%d', b, p, d+1)));
			end
		end
	
		% Generate multiplications
		for p = 0:bitspec.numin-1
			for d = 0:inputdelays(p+1)
				coeff = h(d*bitspec.numin+bitspec.numin-p);
				ksig = ksig + 1;
				comp.arch.signals(ksig) = hdlsignal(sprintf('xp_%d_%d_%d', b, p, d), 1, bitspec.arith.wout);
				kinst = kinst + 1;
				comp.arch.instances(kinst) = hdlinst(cmult, ...
					[hdlmap('win', bitspec.arith.wdata); hdlmap('wout', bitspec.arith.wout); ...
					 hdlmap('c', coeff)], ...
					hdlmap('din', sprintf('x_%d_%d_%d', b, p, d)), ...
					hdlmap('dout', sprintf('xp_%d_%d_%d', b, p, d)));
			end
		end
	
		% Generate accumulation of products
		for p = 0:bitspec.numin-1
			for d = 0:inputdelays(p+1)
				ksig = ksig + 1;
				comp.arch.signals(ksig) = hdlsignal(sprintf('xacc_%d_%d_%d', b, p, d), 1, bitspec.arith.wout);
			end
			for d = 0:inputdelays(p+1)-1
				kinst = kinst + 1;
				comp.arch.instances(kinst) = hdlinst(add, ...
					hdlmap('w', bitspec.arith.wout), ...
					[hdlmap('din1', sprintf('xp_%d_%d_%d', b, p, d)), ...
					 hdlmap('din2', sprintf('xacc_%d_%d_%d', b, p, d+1))], ...
					hdlmap('dout', sprintf('xacc_%d_%d_%d', b, p, d)));
			end
			for d = inputdelays(p+1)
				kass = kass + 1;
				comp.arch.assignments(kass) = hdlass(sprintf('xacc_%d_%d_%d', b, p, d), sprintf('xp_%d_%d_%d', b, p, d));
			end
		end
		for p = 0:bitspec.numin-1
			ksig = ksig + 1;
			comp.arch.signals(ksig) = hdlsignal(sprintf('xacc_%d_%d', b, p), 1, bitspec.arith.wout);
		end
		for p = 0:bitspec.numin-2
			kinst = kinst + 1;
			comp.arch.instances(kinst) = hdlinst(add, ...
					hdlmap('w', bitspec.arith.wout), ...
					[hdlmap('din1', sprintf('xacc_%d_%d_0', b, p)), ...
					 hdlmap('din2', sprintf('xacc_%d_%d', b, p+1))], ...
					hdlmap('dout', sprintf('xacc_%d_%d', b, p)));
		end
		for p = bitspec.numin-1
			kass = kass + 1;
			comp.arch.assignments(kass) = hdlass(sprintf('xacc_%d_%d', b, p), sprintf('xacc_%d_%d_0', b, p));
		end
	
		% Generate output delays
		for d = 0:pipedelay
			ksig = ksig + 1;
			comp.arch.signals(ksig) = hdlsignal(sprintf('yd_%d_%d', b, d), 1, bitspec.arith.wout);
		end
		kass = kass + 1;
		comp.arch.assignments(kass) = hdlass(sprintf('yd_%d_0', b), sprintf('xacc_%d_0', b));
		for d = 0:pipedelay-1
			kinst = kinst + 1;
			comp.arch.instances(kinst) = hdlinst('reg', ...
				hdlmap('wordlength', bitspec.arith.wout), ...
				hdlmap('d', sprintf('yd_%d_%d', b, d)), ...
				hdlmap('q', sprintf('yd_%d_%d', b, d+1)));
		end
	
		% Connect output
		kass = kass + 1;
		comp.arch.assignments(kass) = hdlass(sprintf('y_%d', b), sprintf('yd_%d_%d', b, pipedelay));
	
		hdlspec.components = [hdlspec.components, comp];
	end
	
